//////////////////////////////////////////////////////////////////////////////
// The Magic Library Version 1.0
//
// File: StrClass.h
// Author: Daniel Tschan (d.tschan@switzerland.org)
// Created: 10-13-97
// Last modified: 10-30-97
// Documentation: http://iamexwiwww.unibe.ch/studenten/tschan/TML
//
// This file contains the interface of the CString class. The interface is
// identical to the one of the CString class in MFC.

#ifndef __STRCLASS_H__
#define __STRCLASS_H__

#include <Archive.h>
#include <string.h>
#include <stdio.h>
#include <Decls.h>
#include <ctype.h>

class CString
{
public:
// Construction/Destruction
  CString( );
  CString( const CString& stringSrc );
  CString( char ch, int nRepeat = 1 );
  CString( const char* lpch, int nLength );
  CString( const unsigned char* lpsz );
  CString( const char* lpsz );
  ~CString( );

protected:
  CString( int nLen );

public:
// The String as an Array
  int GetLength( ) const;
  BOOL IsEmpty( ) const;
  void Empty( );
  char GetAt( int nIndex ) const;
  char operator[]( int nIndex ) const;
  void SetAt( int nIndex, char ch );
  operator const char*( ) const;

// Assignment/Concatenation
  const CString& operator=( const CString& rOther );
  const CString& operator=( char ch );
  const CString& operator=( const unsigned char* lpsz );
  const CString& operator=( const char* lpsz );

  friend CString operator+( const CString& string1, const CString& string2 );
  friend CString operator+( const CString& string, char ch );
  friend CString operator+( char ch, const CString& string );
  friend CString operator+( const CString& string, const char* lpsz );
  friend CString operator+( const char* lpsz, const CString& string );

  const CString& operator+=( const CString& string ); 
  const CString& operator+=( char ch );
  const CString& operator+=( const char* lpsz );

// Comparision
  friend BOOL operator==( const CString& s1, const CString& s2 );
  friend BOOL operator==( const CString& s1, const char* s2 );
  friend BOOL operator==( const char* s1, const CString& s2 );
  
  friend BOOL operator!=( const CString& s1, const CString& s2 );
  friend BOOL operator!=( const CString& s1, const char* s2 );
  friend BOOL operator!=( const char* s1, const CString& s2);
  
  friend BOOL operator<( const CString& s1, const CString& s2 );
  friend BOOL operator<( const CString& s1, const char* s2 );
  friend BOOL operator<( const char* s1, const CString& s2 );
  
  friend BOOL operator>( const CString& s1, const CString& s2 );
  friend BOOL operator>( const CString& s1, const char* s2 );
  friend BOOL operator>( const char* s1, const CString& s2 );
  
  friend BOOL operator<=( const CString& s1, const CString& s2 );
  friend BOOL operator<=( const CString& s1, const char* s2 );
  friend BOOL operator<=( const char* s1, const CString& s2 );
  
  friend BOOL operator>=( const CString& s1, const CString& s2 );
  friend BOOL operator>=( const CString& s1, const char* s2 );
  friend BOOL operator>=( const char* s1, const CString& s2 );
  
  int Compare( const char* lpsz ) const; 
  int CompareNoCase( const char* lpsz ) const;
  int Collate( const char* lpsz ) const;

// Extraction
  CString Mid( int nFirst ) const;
  CString Mid( int nFirst, int nCount ) const;
  CString Left( int nCount ) const;
  CString Right( int nCount ) const;
  CString SpanIncluding( const char* lpszCharSet ) const;
  CString SpanExcluding( const char* lpszCharSet ) const;

// Other conversion
  void MakeUpper( );
  void MakeLower( );
  void MakeReverse( );
  void Format( const char* lpszFormat, ... );
  void TrimLeft( );
  void TrimRight( );

// Searching
  int Find( char ch, int nStart = 0 ) const;
  int Find( const char* lpszSub, int nStart = 0 ) const;
  int ReverseFind( char ch ) const;
  int FindOneOf( const char* lpszCharSet ) const;

// Archive
  friend CArchive& operator<<( CArchive& ar, const CString& string );
  friend CArchive& operator>>( CArchive& ar, CString& string );

// Buffer Access
  char* GetBuffer( int nMinBufLength );
  char* GetBufferSetLength( int nMinBufLength );
  void ReleaseBuffer( int nNewLength = -1 );
  void FreeExtra( );
  friend istream& operator>>( istream& rStream, CString& rString );
  
private:
  void NewString( int nLen = 0 );
  void ReallocString( int nLen );
  void ResizeString( int nLen );
  void ForceResizeString( int nLen );

private:
  int m_nLen;
  int m_nAlloc;
  char* m_pszString;
//  static char* m_pszEmpty;
};



// The String as an Array

inline int CString::GetLength( ) const
{
  return m_nLen;
}

inline BOOL CString::IsEmpty( ) const
{
  return m_nLen == 0;
}

inline void CString::Empty( )
{
  ReallocString( 0 );
}

inline char CString::GetAt( int nIndex ) const
{
  ASSERT( nIndex >= 0 && nIndex < m_nLen );
  return m_pszString[ nIndex ];
}

inline char CString::operator[]( int nIndex ) const
{
  ASSERT( nIndex >= 0 && nIndex < m_nLen );
  return GetAt( nIndex );
}

inline void CString::SetAt( int nIndex, char ch )
{
  ASSERT( nIndex >= 0 && nIndex < m_nLen );
  m_pszString[ nIndex ] = ch;
}

inline CString::operator const char*( ) const
{
  return m_pszString;
}



// Other conversions

inline void CString::MakeUpper( )
{
#ifdef _WIN32
  strupr( m_pszString );
#else
  for ( int i = 0; i < m_nLen; i++ )
    m_pszString[ i ] = toupper( ( int ) m_pszString );
#endif
}

inline void CString::MakeLower( )
{
#ifdef _WIN32
  strlwr( m_pszString );
#else
  for ( int i = 0; i < m_nLen; i++ )
    m_pszString[ i ] = tolower( ( int ) m_pszString );
#endif
}

inline void CString::MakeReverse( )
{
#ifdef _WIN32
  strrev( m_pszString );
#else
  char chTemp;

  for ( int i = 0; i < m_nLen / 2; i++ )
  {
    chTemp = m_pszString[ i ];
    m_pszString[ i ] = m_pszString[ m_nLen - 1 - i ];
    m_pszString[ m_nLen - i - 1 ] = chTemp;
  }
#endif
}



// Comparison

inline int CString::Compare( const char* lpsz ) const
{
  return strcmp( m_pszString, lpsz );
}

inline int CString::CompareNoCase( const char* lpsz ) const
{
#ifdef _WIN32
  return stricmp( m_pszString, lpsz );
#else
  return strcasecmp( m_pszString, lpsz );
#endif
}

inline int CString::Collate( const char* lpsz ) const
{
  return strcoll( m_pszString, lpsz );
}

inline BOOL operator==( const CString& s1, const CString& s2 )
  {  return s1.Compare( s2 ) == 0; }
inline BOOL operator==( const CString& s1, const char* s2 )
  { return s1.Compare( s2 ) == 0; }
inline BOOL operator==( const char* s1, const CString& s2 )
  {  return s2.Compare( s1 ) == 0; }
inline BOOL operator!=( const CString& s1, const CString& s2 )
  {  return s1.Compare( s2 ) != 0; }
inline BOOL operator!=( const CString& s1, const char* s2 )
  {  return s1.Compare( s2 ) != 0; }
inline BOOL operator!=(const char* s1, const CString& s2)
  { return s2.Compare( s1 ) != 0; }
inline BOOL operator<( const CString& s1, const CString& s2 )
  { return s1.Compare( s2 ) < 0; }
inline BOOL operator<( const CString& s1, const char* s2 )
  { return s1.Compare( s2 ) < 0; }
inline BOOL operator<( const char* s1, const CString& s2 )
  { return s2.Compare( s1 ) > 0; }
inline BOOL operator>( const CString& s1, const CString& s2 )
  { return s1.Compare( s2 ) > 0; }
inline BOOL operator>( const CString& s1, const char* s2 )
  { return s1.Compare( s2 ) > 0; }
inline BOOL operator>( const char* s1, const CString& s2 )
  { return s2.Compare( s1 ) < 0; }
inline BOOL operator<=( const CString& s1, const CString& s2 )
  { return s1.Compare( s2 ) <= 0; }
inline BOOL operator<=( const CString& s1, const char* s2 )
  { return s1.Compare( s2 ) <= 0; }
inline BOOL operator<=( const char* s1, const CString& s2 )
  { return s2.Compare( s1 ) >= 0; }
inline BOOL operator>=( const CString& s1, const CString& s2 )
  { return s1.Compare( s2 ) >= 0; }
inline BOOL operator>=( const CString& s1, const char* s2 )
  { return s1.Compare( s2 ) >= 0; }
inline BOOL operator>=( const char* s1, const CString& s2 )
  { return s2.Compare( s1 ) <= 0; }



// Extraction

inline CString CString::Mid( int nFirst ) const
{
  ASSERT( nFirst >= 0 && nFirst < m_nLen || ( nFirst == 0 && m_nLen == 0 ) );

  return CString( m_pszString + nFirst );
}

inline CString CString::Mid( int nFirst, int nCount ) const
{
  ASSERT( nFirst >= 0 && nFirst < m_nLen );
  ASSERT( nFirst + nCount <= m_nLen );

  return CString( m_pszString + nFirst, nCount );
}

inline CString CString::Left( int nCount ) const
{
  ASSERT( nCount >= 0 && nCount <= m_nLen );

  return CString( m_pszString, nCount );
}

inline CString CString::Right( int nCount ) const
{
  ASSERT( nCount >= 0 && nCount < m_nLen );

  return CString( m_pszString + m_nLen - nCount, nCount );
}

inline CString CString::SpanIncluding( const char* lpszCharSet ) const
{
  return Left( strspn( m_pszString, lpszCharSet) );
}

inline CString CString::SpanExcluding( const char* lpszCharSet ) const
{
  return Left( strcspn( m_pszString, lpszCharSet ) );
}



// Buffer Access

inline void CString::FreeExtra( )
{
  ForceResizeString( m_nLen );
}

#endif // __STRCLASS_H__
